'use strict';

import "isomorphic-fetch"
//https://github.com/gulpjs/gulp/blob/4.0/docs/API.md#gulpsrcglobs-options
//https://markgoodyear.com/2015/06/using-es6-with-gulp/
//https://stackoverflow.com/questions/36484156/es6-fetch-is-undefined

// Basic packages.
let del          = require('del');
let gulp         = require('gulp');
let autoprefixer = require('autoprefixer');
let browserSync  = require('browser-sync').create();

// Regular packages.
let sass         = require('gulp-sass');
let concat       = require('gulp-concat');
let uglify       = require('gulp-uglify');
let imagemin     = require('gulp-imagemin');
let cleanCSS     = require('gulp-clean-css');
let sourcemaps   = require('gulp-sourcemaps');
let plumber      = require('gulp-plumber');

/**
 * GULP TASKS & WATCHERS.
 */

// Main build tasks.
gulp.task('default', _watcher);
gulp.task('production', gulp.series(_clean, gulp.parallel(
    _handleSCSS, 
    _handleProductionSCSS,
    _handleScripts, 
    _handleProductionScripts,
    _handleImages
), _productionPackage, _jobDone));

// Single tasks.
gulp.task('clean', _clean);
gulp.task('scss', _handleSCSS);
gulp.task('js', _handleScripts);
gulp.task('images', _handleImages);

// Production tasks.
gulp.task('production-scss', _handleProductionSCSS);
gulp.task('production-js', _handleProductionScripts);
gulp.task('production-package', _productionPackage);

gulp.task('job', _jobDone);
gulp.task('server', _startServer);
/**
 * GULP ACTION FUNCTIONS.
 */

// Define the watcher functions which run by default
function _watcher(done) {
    gulp.watch('src/js/**/*.js', gulp.parallel(_handleScripts, _handleProductionScripts));
    // gulp.watch('src/scss/flightdeck/**/*.scss', gulp.series(_handleFlightDeck))
    gulp.watch('src/scss/tailwind/tailwind.js', gulp.series(_handleTailwind))
    gulp.watch('src/scss/**/*.scss', gulp.parallel(_handleSCSS, _handleProductionSCSS));
    gulp.watch('src/img/**/*', gulp.series(_handleImages))
    done();
}

// Clean the build directory for a new fresh build
function _clean(done) {
    del(['dist/']);
    done();
}

// Handle the proccessing of SCSS
function _handleSCSS(done) {
    gulp.src('src/scss/*.scss')
        .pipe(plumber())
        .pipe(sourcemaps.init())
        .pipe(sass())
        .pipe(concat('styles.css'))
        .pipe(sourcemaps.write())
        .pipe(gulp.dest('dist/css'));
    done();
}

// Handle the proccessing of SCSS framework
function _handleFlightDeck(done) {
    gulp.src('src/scss/flightdeck/*.scss')
        .pipe(plumber())
        .pipe(sourcemaps.init())
        .pipe(sass())
        .pipe(concat('flightdeck.css'))
        .pipe(sourcemaps.write())
        .pipe(gulp.dest('dist/css'));
    done();
}

// Handle the proccessing of SCSS for production with compression
function _handleProductionSCSS(done) {
    gulp.src('src/scss/*.scss')
        .pipe(plumber())
        .pipe(sourcemaps.init())
        .pipe(sass({outputStyle: 'compressed'}))
        .pipe(concat('styles.min.css'))
        .pipe(cleanCSS())
        .pipe(sourcemaps.write())
        .pipe(gulp.dest('dist/css/'));
    done();
}

// Handle the proccessing of any JavaScript files
function _handleScripts(done) {
    gulp.src('src/js/*.js')
        .pipe(plumber())
        .pipe(sourcemaps.init())
        .pipe(concat('app.js'))
        .pipe(sourcemaps.write())
        .pipe(gulp.dest('dist/js'));
    done();    
}

// Handle the proccessing of any JavaScript files with compression
function _handleProductionScripts(done) {
    gulp.src('src/js/*.js')
        .pipe(plumber())
        .pipe(sourcemaps.init())
        .pipe(concat('app.min.js'))
        .pipe(uglify())
        .pipe(sourcemaps.write())
        .pipe(gulp.dest('dist/js'));
    done();
}

// Handle the build of the Tailwind framework
function _handleTailwind(done) {
    gulp.src('src/scss/tailwind/tailwind.css')
        .pipe(postcss([
            tailwindcss('src/scss/tailwind/tailwind.js'),
            require('autoprefixer'),
        ]))
        .pipe(gulp.dest('dist/css'));
    done();    
}

// Handle image minification for production
function _handleImages(done) {
    gulp.src('src/img/*')
        .pipe(plumber())
        .pipe(imagemin([
            imagemin.gifsicle({interlaced: true}),
            imagemin.jpegtran({progressive: true}),
            imagemin.optipng({optimizationLevel: 5}),
            imagemin.svgo({
                plugins: [
                    {removeViewBox: true},
                    {cleanupIDs: false}
                ]
            })
        ]))
        .pipe(gulp.dest('dist/img'));
    done();
}

// Handle the CSS file merge for production
function _productionPackage(done) {
    gulp.src(['dist/css/styles.min.css'])
        .pipe(concat('app.min.css'))
        .pipe(postcss([autoprefixer( {
            cascade: false,
            add: true,
            remove: true,
            supports: true,
            flexbox: true,
        })]))
        .pipe(cleanCSS())
        .pipe(gulp.dest('dist/css'));
    done();
}

// A silly little funciton to prevent the requirement of an anonymous function. From: https://quotes.rest/#!/qod/get_qod
function _jobDone(done) {
    fetch("https://quotes.rest/qod")
        .then((resp) => resp.json())
        .then( function(data) {
            if (data.success == 200) {           
                console.log('[What the quote] Your job is done so enjoy this: "', data.contents.quotes[0].quote + " by " + data.contents.quotes[0].author + '"');
            } else {
                console.log('[What the quote] Your job is done but no quote could be loaded. Sorry!');
            }
        })
    done();
}